/**
 * Popup logic: auth check, scrape trigger, history display.
 * Scraping runs in background.js so popup can close without interrupting.
 */

const $ = (sel) => document.querySelector(sel);

function show(id) {
  document.querySelectorAll('.section').forEach((el) => (el.style.display = 'none'));
  $(id).style.display = 'block';
}

async function sendMessage(msg) {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage(msg, resolve);
  });
}

let pollTimer = null;

function showState(state) {
  if (state.status === 'navigating') {
    $('#btn-scrape').style.display = 'none';
    $('#scrape-warning').style.display = 'block';
    $('#scrape-progress').style.display = 'block';
    $('#scrape-result').style.display = 'none';
    $('#scrape-error').style.display = 'none';
    $('#progress-text').textContent = 'Opening activity page...';
  } else if (state.status === 'scraping' || state.status === 'saving') {
    $('#btn-scrape').style.display = 'none';
    $('#scrape-warning').style.display = 'block';
    $('#scrape-progress').style.display = 'block';
    $('#scrape-result').style.display = 'none';
    $('#scrape-error').style.display = 'none';
    if (state.status === 'saving') {
      $('#progress-text').textContent = `Saving ${state.found} posts to tuls.me...`;
    } else {
      const found = state.found || 0;
      const target = state.target || '?';
      $('#progress-text').textContent = `${found} / ${target} posts`;
    }
  } else if (state.status === 'done') {
    $('#scrape-progress').style.display = 'none';
    $('#scrape-warning').style.display = 'none';
    $('#scrape-result').style.display = 'block';
    const d = state.result;
    $('#result-text').textContent = `Done! ${d.created} new, ${d.updated} updated.`;
    $('#btn-scrape').style.display = 'block';
    loadHistory();
    stopPolling();
  } else if (state.status === 'error') {
    $('#scrape-progress').style.display = 'none';
    $('#scrape-warning').style.display = 'none';
    $('#scrape-error').style.display = 'block';
    $('#error-text').textContent = state.error;
    $('#btn-scrape').style.display = 'block';
    stopPolling();
  }
}

function startPolling() {
  if (pollTimer) return;
  pollTimer = setInterval(async () => {
    const state = await sendMessage({ type: 'GET_SCRAPE_STATE' });
    if (state) showState(state);
  }, 1000);
}

function stopPolling() {
  if (pollTimer) {
    clearInterval(pollTimer);
    pollTimer = null;
  }
}

/** Normalize a LinkedIn profile URL to its /recent-activity/all/ page. */
function toActivityUrl(input) {
  try {
    const url = new URL(input.trim());
    if (url.hostname !== 'www.linkedin.com') return null;
    // Extract /in/username from the path
    const match = url.pathname.match(/^\/in\/([^/]+)/);
    if (!match) return null;
    return `https://www.linkedin.com/in/${match[1]}/recent-activity/all/`;
  } catch {
    return null;
  }
}

async function init() {
  // Check auth
  const auth = await sendMessage({ type: 'CHECK_AUTH' });
  if (!auth || !auth.authenticated) {
    show('#auth-prompt');
    return;
  }

  show('#scrape-ready');

  // Pre-fill URL from current tab if it's LinkedIn
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (tab && tab.url) {
    try {
      const url = new URL(tab.url);
      if (url.hostname === 'www.linkedin.com' && url.pathname.match(/^\/in\//)) {
        $('#profile-url').value = tab.url;
      }
    } catch {}
  }

  // Check if a scrape is already running
  const state = await sendMessage({ type: 'GET_SCRAPE_STATE' });
  if (state && (state.status === 'scraping' || state.status === 'saving' || state.status === 'navigating')) {
    showState(state);
    startPolling();
    loadHistory();
    return;
  }

  // If last scrape had a result, show it
  if (state && state.status === 'done') showState(state);

  // Scrape button
  $('#btn-scrape').addEventListener('click', async () => {
    const profileInput = $('#profile-url').value;
    const activityUrl = toActivityUrl(profileInput);
    if (!activityUrl) {
      $('#scrape-error').style.display = 'block';
      $('#error-text').textContent = 'Enter a valid LinkedIn profile URL (linkedin.com/in/...)';
      return;
    }
    $('#scrape-error').style.display = 'none';

    const maxPosts = parseInt($('#max-posts').value, 10) || 5;
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
    const tabId = activeTab.id;

    await sendMessage({ type: 'START_SCRAPE_BG', tabId, maxPosts, activityUrl });
    showState({ status: 'navigating' });
    startPolling();
  });

  loadHistory();
}

async function loadHistory() {
  const resp = await sendMessage({ type: 'GET_POSTS' });
  if (!resp || !resp.ok || !resp.data || resp.data.length === 0) return;

  $('#history').style.display = 'block';
  const list = $('#history-list');
  list.innerHTML = '';

  const total = resp.data.length;
  const item = document.createElement('div');
  item.className = 'history-item';
  item.innerHTML = `<span class="history-count">${total} posts saved</span>`;
  list.appendChild(item);
}

document.addEventListener('DOMContentLoaded', init);
