/**
 * Background service worker: auth, scraping orchestration, API calls to tuls.me
 */

const API_BASE = 'https://tuls.me';

// Scrape state — persists even if popup closes
let scrapeState = { status: 'idle', result: null, error: null, found: 0, target: 0 };

async function getAuthToken() {
  return new Promise((resolve) => {
    chrome.cookies.get({ url: API_BASE, name: 'auth-token' }, (cookie) => {
      resolve(cookie ? cookie.value : null);
    });
  });
}

async function apiCall(path, options = {}) {
  const token = await getAuthToken();
  if (!token) {
    return { error: 'Not authenticated', status: 401 };
  }

  const url = `${API_BASE}${path}`;
  const headers = {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json',
    ...options.headers,
  };

  try {
    const resp = await fetch(url, { ...options, headers });
    const data = resp.status === 204 ? null : await resp.json();
    return { data, status: resp.status, ok: resp.ok };
  } catch (e) {
    return { error: e.message, status: 0 };
  }
}

async function ensureContentScript(tabId) {
  try {
    await chrome.tabs.sendMessage(tabId, { type: 'PING' });
  } catch (e) {
    await chrome.scripting.executeScript({
      target: { tabId },
      files: ['content/scraper.js'],
    });
  }
}

/** Wait for a tab to finish loading. */
function waitForTabLoad(tabId) {
  return new Promise((resolve) => {
    const listener = (id, info) => {
      if (id === tabId && info.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(listener);
        resolve();
      }
    };
    chrome.tabs.onUpdated.addListener(listener);
  });
}

async function runScrape(tabId, maxPosts, activityUrl) {
  scrapeState = { status: 'navigating', result: null, error: null, found: 0, target: maxPosts };

  try {
    // Navigate to activity page
    chrome.tabs.update(tabId, { url: activityUrl });
    await waitForTabLoad(tabId);
    // Small delay for LinkedIn SPA to settle
    await new Promise((r) => setTimeout(r, 2000));

    scrapeState.status = 'scraping';

    await ensureContentScript(tabId);

    // Ask content script to scrape
    const result = await chrome.tabs.sendMessage(tabId, { type: 'START_SCRAPE', maxPosts });

    if (!result || !result.posts || result.posts.length === 0) {
      scrapeState = { status: 'error', result: null, error: 'No posts found on the page.', found: 0, target: maxPosts };
      return;
    }

    scrapeState.status = 'saving';
    scrapeState.found = result.posts.length;

    // Submit to backend
    const resp = await apiCall('/api/linkedin/posts', {
      method: 'POST',
      body: JSON.stringify({
        profile_url: result.profileUrl,
        posts: result.posts,
      }),
    });

    if (resp.ok) {
      scrapeState = { status: 'done', result: resp.data, error: null, found: result.posts.length, target: maxPosts };
    } else {
      scrapeState = { status: 'error', result: null, error: resp.data?.error || 'Failed to save.', found: 0, target: maxPosts };
    }
  } catch (e) {
    scrapeState = { status: 'error', result: null, error: e.message, found: 0, target: maxPosts };
  }
}

// Message handlers
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.type === 'SCRAPE_PROGRESS') {
    scrapeState.found = msg.found;
    scrapeState.target = msg.target;
    return false;
  }

  if (msg.type === 'CHECK_AUTH') {
    getAuthToken().then((token) => {
      sendResponse({ authenticated: !!token });
    });
    return true;
  }

  if (msg.type === 'START_SCRAPE_BG') {
    runScrape(msg.tabId, msg.maxPosts, msg.activityUrl);
    sendResponse({ ok: true });
    return false;
  }

  if (msg.type === 'GET_SCRAPE_STATE') {
    sendResponse(scrapeState);
    return false;
  }

  if (msg.type === 'GET_POSTS') {
    apiCall('/api/linkedin/posts').then(sendResponse);
    return true;
  }
});
